package fsm.chath;

import java.io.*;
import java.util.*;
import java.net.*;
import java.awt.*;
import java.awt.event.*;

import simple.http.*;
import simple.http.connect.*;
import simple.http.load.*;
import simple.http.serve.*;
import simple.util.net.*;

public class ChaTh extends BasicService {
    private static final int SYNC_MODE = 0;
    private static final int MULTI_MODE = 1;
    private static final int SINGLE_MODE = 2;
    
    private static String nick = "";
    private static String peer = "";
    private static int mode;
    
    private ChaTh.ChaThFrame frame;
    private Dispatcher dispatcher;
    
    /*
     * This inner class is responsible for message dispatching.
     * It sends a message and 
     */
    public class Dispatcher extends Thread {
        String message;
        
        public Dispatcher() {
            start();
        }
        
        public void run() {
            try {
                while(true) {
                    if (message != null) {
                        doSend(message);
                    }
                    synchronized (this) {
                        wait();
                    }
                }
            }
            catch(InterruptedException ie) {
                ie.printStackTrace();
            }
        }
        
        public void sendMessage(String message) {
            this.message = message;
            synchronized(this) {
                notify();
            }
        }
        
    }
    
    private class ChaThFrame extends Frame implements ActionListener, WindowListener {
        private TextField ebox;
        private TextArea tbox;
        private Button bsend;
        
        public ChaThFrame() {
            init();
        }
        
        private void init() {
            Panel panel = new Panel(new BorderLayout());
            setTitle("ChaTh");
            ebox = new TextField();
            tbox = new TextArea();
            bsend = new Button("Send");
            setLayout(new BorderLayout());
            
            panel.add(ebox, BorderLayout.CENTER);
            panel.add(bsend, BorderLayout.EAST);
            add(tbox, BorderLayout.CENTER);
            add(panel, BorderLayout.SOUTH);
            bsend.addActionListener(this);
            ebox.addActionListener(this);
            addWindowListener(this);
            setSize(450, 200);
        }
        
        public void append(String nick, String message) {
            tbox.append(nick + ": " + message + "\n");
        }
        
        public void actionPerformed(ActionEvent e) {
            if (e.getSource() == bsend || e.getSource() == ebox) {
                switch(mode) {
                    case SYNC_MODE:
                        doSyncSend(ebox.getText());
                        break;
                    case MULTI_MODE:
                        doMultiThreadSend(ebox.getText());
                        break;
                    case SINGLE_MODE:
                        doSingleThreadSend(ebox.getText());
                        break;
                }
                append("me", ebox.getText());
                ebox.setText("");
            }
        }
        
        public void windowActivated(WindowEvent e) {}
        
        public void windowClosed(WindowEvent e) {}
        
        public void windowClosing(WindowEvent e) {
            System.exit(0);
        }
        
        public void windowDeactivated(WindowEvent e) {}
        
        public void windowDeiconified(WindowEvent e) {}
        
        public void windowIconified(WindowEvent e) {}
        
        public void windowOpened(WindowEvent e) {}
        
    }
    
    public ChaTh(Context context) {
        super(context);
        frame = new ChaTh.ChaThFrame();
        frame.setVisible(true);
    }
    
    
    /*
     * doSyncSend, doMultiSend and doSingleSend perform
     * message dispatching:
     *
     * doSyncSend invokes doSend, therefore it is being blocked
     * until doSend returns
     *
     * doMultiSend creates a new anonymous thread each time
     * it is invoked. Such a thread invokes doSend in its run method.
     * doMultiSend returns immediatly.
     *
     * doSingleSend creates and starts a Dispatcher thread (if
     * needed) and uses it to send messages. doSingleSend
     * returns immediatly.
     *
     */
    
    private void doSyncSend(String message) {
        doSend(message);
    }
    
    private void doSingleThreadSend(String message) {
        if (dispatcher == null) {
            dispatcher = new Dispatcher();
        }
        dispatcher.sendMessage(message);
    }
    
    private void doMultiThreadSend(String message) {
        final String msg = message;
        Thread t = new Thread() {
            public void run() {
                doSyncSend(msg);
            }
        };
        t.start();
    }
    
    /*
     * doSend performs actual message dispatching: it creates the Http
     * connection and fill required parameters.
     *
     */
    private void doSend(String message) {
        try {
            URL url = new URL("http://" + peer + ":1080/message");
            HttpURLConnection connection = (HttpURLConnection) url.openConnection();
            connection.setRequestMethod("POST");
            String data = URLEncoder.encode("nick", "UTF-8") + "=" + URLEncoder.encode(nick, "UTF-8");
            data += "&" + URLEncoder.encode("message", "UTF-8") + "=" + URLEncoder.encode(message, "UTF-8");
            connection.setDoOutput(true);
            connection.connect();
            OutputStreamWriter writer = new OutputStreamWriter(connection.getOutputStream());
            writer.write(data);
            writer.flush();
            if (connection.getResponseCode() != HttpURLConnection.HTTP_OK) {
                throw new Exception();
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    private void doReceive(String nick, String message) {
        frame.append(nick, message);
    }
    
    /*
     * The process method is required by the BasicService
     * public interface and it is invoked by the Simple web
     * server engine web to manage incoming connections.
     */
    public void process(Request req, Response resp) throws Exception {
        doReceive(req.getParameter("nick"), req.getParameter("message"));
        resp.commit();
        resp.getOutputStream().close();
    }
    
    public static void main(String[] args) throws Exception {
        if (args.length == 0) {
            System.out.println("Usage: ChaTh nick peer mode");
            System.exit(1);
        }
        else {
            nick = args[0];
            peer = args[1];
            if (args[2].equalsIgnoreCase("sync")) mode = SYNC_MODE;
            else if (args[2].equalsIgnoreCase("multi")) mode = MULTI_MODE;
            else if (args[2].equalsIgnoreCase("single")) mode = SINGLE_MODE;
        }
        try {
            LoaderEngine engine = new LoaderEngine();
            engine.load("chat", "fsm.chath.ChaTh");
            engine.link("/message", "chat");
            ProtocolHandler handler = HandlerFactory.getInstance(engine);
            Connection connection = ConnectionFactory.getConnection(handler);
            connection.connect(new ServerSocket(1080));
        }
        catch(Exception e) {
            e.printStackTrace();
        }
    }
    
}